/*
 * Decompiled with CFR 0.152.
 */
package io.netty5.handler.codec.compression;

import io.netty5.buffer.Buffer;
import io.netty5.buffer.BufferAllocator;
import io.netty5.buffer.BufferComponent;
import io.netty5.buffer.ComponentIterator;
import io.netty5.handler.codec.compression.CompressionException;
import io.netty5.handler.codec.compression.Compressor;
import io.netty5.handler.codec.compression.ZlibWrapper;
import io.netty5.util.internal.EmptyArrays;
import io.netty5.util.internal.ObjectUtil;
import io.netty5.util.internal.SystemPropertyUtil;
import io.netty5.util.internal.logging.InternalLogger;
import io.netty5.util.internal.logging.InternalLoggerFactory;
import java.nio.ByteBuffer;
import java.util.Objects;
import java.util.function.Supplier;
import java.util.zip.CRC32;
import java.util.zip.Deflater;

public final class ZlibCompressor
implements Compressor {
    private static final InternalLogger logger = InternalLoggerFactory.getInstance(ZlibCompressor.class);
    private static final int MAX_INITIAL_OUTPUT_BUFFER_SIZE;
    private final ZlibWrapper wrapper;
    private final Deflater deflater;
    private final CRC32 crc = new CRC32();
    private static final byte[] gzipHeader;
    private State state = State.PROCESSING;
    private boolean writeHeader = true;

    private ZlibCompressor(ZlibWrapper wrapper, int compressionLevel) {
        this.wrapper = wrapper;
        this.deflater = new Deflater(compressionLevel, wrapper != ZlibWrapper.ZLIB);
    }

    private ZlibCompressor(int compressionLevel, byte[] dictionary) {
        this.wrapper = ZlibWrapper.ZLIB;
        this.deflater = new Deflater(compressionLevel);
        this.deflater.setDictionary(dictionary);
    }

    public static Supplier<ZlibCompressor> newFactory() {
        return ZlibCompressor.newFactory(6);
    }

    public static Supplier<ZlibCompressor> newFactory(int compressionLevel) {
        return ZlibCompressor.newFactory(ZlibWrapper.ZLIB, compressionLevel);
    }

    public static Supplier<ZlibCompressor> newFactory(ZlibWrapper wrapper) {
        return ZlibCompressor.newFactory(wrapper, 6);
    }

    public static Supplier<ZlibCompressor> newFactory(ZlibWrapper wrapper, int compressionLevel) {
        ObjectUtil.checkInRange(compressionLevel, 0, 9, "compressionLevel");
        Objects.requireNonNull(wrapper, "wrapper");
        if (wrapper == ZlibWrapper.ZLIB_OR_NONE) {
            throw new IllegalArgumentException("wrapper '" + ZlibWrapper.ZLIB_OR_NONE + "' is not allowed for compression.");
        }
        return () -> new ZlibCompressor(wrapper, compressionLevel);
    }

    public static Supplier<ZlibCompressor> newFactory(byte[] dictionary) {
        return ZlibCompressor.newFactory(6, dictionary);
    }

    public static Supplier<ZlibCompressor> newFactory(int compressionLevel, byte[] dictionary) {
        if (compressionLevel < 0 || compressionLevel > 9) {
            throw new IllegalArgumentException("compressionLevel: " + compressionLevel + " (expected: 0-9)");
        }
        Objects.requireNonNull(dictionary, "dictionary");
        return () -> new ZlibCompressor(compressionLevel, dictionary);
    }

    @Override
    public Buffer compress(Buffer uncompressed, BufferAllocator allocator) throws CompressionException {
        switch (this.state) {
            case CLOSED: {
                throw new CompressionException("Compressor closed");
            }
            case FINISHED: {
                return allocator.allocate(0);
            }
            case PROCESSING: {
                return this.compressData(uncompressed, allocator);
            }
        }
        throw new IllegalStateException();
    }

    private Buffer compressData(Buffer uncompressed, BufferAllocator allocator) {
        int len = uncompressed.readableBytes();
        if (len == 0) {
            return allocator.allocate(0);
        }
        int sizeEstimate = (int)Math.ceil((double)len * 1.001) + 12;
        if (this.writeHeader) {
            switch (this.wrapper) {
                case GZIP: {
                    sizeEstimate += gzipHeader.length;
                    break;
                }
                case ZLIB: {
                    sizeEstimate += 2;
                    break;
                }
            }
        }
        if (sizeEstimate < 0 || sizeEstimate > MAX_INITIAL_OUTPUT_BUFFER_SIZE) {
            sizeEstimate = MAX_INITIAL_OUTPUT_BUFFER_SIZE;
        }
        Buffer out = allocator.allocate(sizeEstimate);
        try (ComponentIterator readableIteration = uncompressed.forEachComponent();){
            BufferComponent readableComponent = (BufferComponent)readableIteration.firstReadable();
            while (readableComponent != null) {
                this.compressData(readableComponent.readableBuffer(), out);
                readableComponent = (BufferComponent)((ComponentIterator.Next)((Object)readableComponent)).nextReadable();
            }
        }
        return out;
    }

    private void compressData(ByteBuffer in, Buffer out) {
        try {
            if (this.writeHeader) {
                this.writeHeader = false;
                if (this.wrapper == ZlibWrapper.GZIP) {
                    out.writeBytes(gzipHeader);
                }
            }
            if (this.wrapper == ZlibWrapper.GZIP) {
                int position = in.position();
                this.crc.update(in);
                in.position(position);
            }
            this.deflater.setInput(in);
            while (true) {
                this.deflate(out);
                if (out.writableBytes() == 0) {
                    out.ensureWritable(out.writerOffset());
                    continue;
                }
                if (this.deflater.needsInput()) break;
            }
            this.deflater.setInput(EmptyArrays.EMPTY_BYTES);
        }
        catch (Throwable cause) {
            out.close();
            throw cause;
        }
    }

    @Override
    public Buffer finish(BufferAllocator allocator) {
        switch (this.state) {
            case CLOSED: {
                throw new CompressionException("Compressor closed");
            }
            case FINISHED: 
            case PROCESSING: {
                this.state = State.FINISHED;
                Buffer footer = allocator.allocate(256);
                try {
                    if (this.writeHeader && this.wrapper == ZlibWrapper.GZIP) {
                        this.writeHeader = false;
                        footer.writeBytes(gzipHeader);
                    }
                    this.deflater.finish();
                    while (!this.deflater.finished()) {
                        this.deflate(footer);
                    }
                    if (this.wrapper == ZlibWrapper.GZIP) {
                        int crcValue = (int)this.crc.getValue();
                        int uncBytes = this.deflater.getTotalIn();
                        footer.writeByte((byte)crcValue);
                        footer.writeByte((byte)(crcValue >>> 8));
                        footer.writeByte((byte)(crcValue >>> 16));
                        footer.writeByte((byte)(crcValue >>> 24));
                        footer.writeByte((byte)uncBytes);
                        footer.writeByte((byte)(uncBytes >>> 8));
                        footer.writeByte((byte)(uncBytes >>> 16));
                        footer.writeByte((byte)(uncBytes >>> 24));
                    }
                    this.deflater.end();
                    return footer;
                }
                catch (Throwable cause) {
                    footer.close();
                    throw cause;
                }
            }
        }
        throw new IllegalStateException();
    }

    @Override
    public boolean isFinished() {
        return this.state != State.PROCESSING;
    }

    @Override
    public boolean isClosed() {
        return this.state == State.CLOSED;
    }

    @Override
    public void close() {
        if (this.state == State.PROCESSING) {
            this.deflater.end();
        }
        this.state = State.CLOSED;
    }

    private void deflate(Buffer out) {
        try (ComponentIterator writableIteration = out.forEachComponent();){
            BufferComponent writableComponent = (BufferComponent)writableIteration.firstWritable();
            while (writableComponent != null) {
                int numBytes;
                if (writableComponent.hasWritableArray()) {
                    while ((numBytes = this.deflater.deflate(writableComponent.writableArray(), writableComponent.writableArrayOffset(), writableComponent.writableBytes(), 2)) > 0) {
                        writableComponent.skipWritableBytes(numBytes);
                    }
                } else {
                    while ((numBytes = this.deflater.deflate(writableComponent.writableBuffer(), 2)) > 0) {
                        writableComponent.skipWritableBytes(numBytes);
                    }
                }
                writableComponent = (BufferComponent)((ComponentIterator.Next)((Object)writableComponent)).nextWritable();
            }
        }
    }

    static {
        gzipHeader = new byte[]{31, -117, 8, 0, 0, 0, 0, 0, 0, 0};
        MAX_INITIAL_OUTPUT_BUFFER_SIZE = SystemPropertyUtil.getInt("io.netty.jdkzlib.encoder.maxInitialOutputBufferSize", 65536);
        if (logger.isDebugEnabled()) {
            logger.debug("-Dio.netty.jdkzlib.encoder.maxInitialOutputBufferSize={}", (Object)MAX_INITIAL_OUTPUT_BUFFER_SIZE);
        }
    }

    private static enum State {
        PROCESSING,
        FINISHED,
        CLOSED;

    }
}

