/*
 * Decompiled with CFR 0.152.
 */
package io.netty5.handler.codec.compression;

import io.netty5.buffer.Buffer;
import io.netty5.buffer.BufferAllocator;
import io.netty5.buffer.BufferInputStream;
import io.netty5.buffer.BufferOutputStream;
import io.netty5.handler.codec.compression.CompressionException;
import io.netty5.handler.codec.compression.Compressor;
import io.netty5.util.internal.logging.InternalLogger;
import io.netty5.util.internal.logging.InternalLoggerFactory;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.function.Supplier;
import lzma.sdk.lzma.Encoder;

public final class LzmaCompressor
implements Compressor {
    private static final InternalLogger logger = InternalLoggerFactory.getInstance(LzmaCompressor.class);
    private static final int MEDIUM_DICTIONARY_SIZE = 65536;
    private static final int MIN_FAST_BYTES = 5;
    private static final int MEDIUM_FAST_BYTES = 32;
    private static final int MAX_FAST_BYTES = 273;
    private static final int DEFAULT_MATCH_FINDER = 1;
    private static final int DEFAULT_LC = 3;
    private static final int DEFAULT_LP = 0;
    private static final int DEFAULT_PB = 2;
    private final Encoder encoder;
    private final byte properties;
    private final int littleEndianDictionarySize;
    private static boolean warningLogged;
    private State state = State.PROCESSING;

    private LzmaCompressor(int lc, int lp, int pb, int dictionarySize, boolean endMarkerMode, int numFastBytes) {
        this.encoder = new Encoder();
        this.encoder.setDictionarySize(dictionarySize);
        this.encoder.setEndMarkerMode(endMarkerMode);
        this.encoder.setMatchFinder(1);
        this.encoder.setNumFastBytes(numFastBytes);
        this.encoder.setLcLpPb(lc, lp, pb);
        this.properties = (byte)((pb * 5 + lp) * 9 + lc);
        this.littleEndianDictionarySize = Integer.reverseBytes(dictionarySize);
    }

    public static Supplier<LzmaCompressor> newFactory() {
        return LzmaCompressor.newFactory(65536);
    }

    public static Supplier<LzmaCompressor> newFactory(int lc, int lp, int pb) {
        return LzmaCompressor.newFactory(lc, lp, pb, 65536);
    }

    public static Supplier<LzmaCompressor> newFactory(int dictionarySize) {
        return LzmaCompressor.newFactory(3, 0, 2, dictionarySize);
    }

    public static Supplier<LzmaCompressor> newFactory(int lc, int lp, int pb, int dictionarySize) {
        return LzmaCompressor.newFactory(lc, lp, pb, dictionarySize, false, 32);
    }

    public static Supplier<LzmaCompressor> newFactory(int lc, int lp, int pb, int dictionarySize, boolean endMarkerMode, int numFastBytes) {
        if (lc < 0 || lc > 8) {
            throw new IllegalArgumentException("lc: " + lc + " (expected: 0-8)");
        }
        if (lp < 0 || lp > 4) {
            throw new IllegalArgumentException("lp: " + lp + " (expected: 0-4)");
        }
        if (pb < 0 || pb > 4) {
            throw new IllegalArgumentException("pb: " + pb + " (expected: 0-4)");
        }
        if (lc + lp > 4 && !warningLogged) {
            logger.warn("The latest versions of LZMA libraries (for example, XZ Utils) has an additional requirement: lc + lp <= 4. Data which don't follow this requirement cannot be decompressed with this libraries.");
            warningLogged = true;
        }
        if (dictionarySize < 0) {
            throw new IllegalArgumentException("dictionarySize: " + dictionarySize + " (expected: 0+)");
        }
        if (numFastBytes < 5 || numFastBytes > 273) {
            throw new IllegalArgumentException(String.format("numFastBytes: %d (expected: %d-%d)", numFastBytes, 5, 273));
        }
        return () -> new LzmaCompressor(lc, lp, pb, dictionarySize, endMarkerMode, numFastBytes);
    }

    @Override
    public Buffer compress(Buffer in, BufferAllocator allocator) throws CompressionException {
        switch (this.state) {
            case CLOSED: {
                throw new CompressionException("Compressor closed");
            }
            case FINISHED: {
                return allocator.allocate(0);
            }
            case PROCESSING: {
                int length = in.readableBytes();
                try (Buffer out = LzmaCompressor.allocateBuffer(in, allocator);
                     BufferInputStream bbIn = new BufferInputStream(in.send());
                     BufferOutputStream bbOut = new BufferOutputStream(out);){
                    bbOut.writeByte(this.properties);
                    bbOut.writeInt(this.littleEndianDictionarySize);
                    bbOut.writeLong(Long.reverseBytes(length));
                    this.encoder.code((InputStream)bbIn, (OutputStream)bbOut, -1L, -1L, null);
                }
                return out;
            }
        }
        throw new IllegalStateException();
    }

    private static Buffer allocateBuffer(Buffer in, BufferAllocator allocator) {
        int length = in.readableBytes();
        int maxOutputLength = LzmaCompressor.maxOutputBufferLength(length);
        return allocator.allocate(maxOutputLength);
    }

    private static int maxOutputBufferLength(int inputLength) {
        double factor = inputLength < 200 ? 1.5 : (inputLength < 500 ? 1.2 : (inputLength < 1000 ? 1.1 : (inputLength < 10000 ? 1.05 : 1.02)));
        return 13 + (int)((double)inputLength * factor);
    }

    @Override
    public Buffer finish(BufferAllocator allocator) {
        switch (this.state) {
            case CLOSED: {
                throw new CompressionException("Compressor closed");
            }
            case FINISHED: 
            case PROCESSING: {
                this.state = State.FINISHED;
                return allocator.allocate(0);
            }
        }
        throw new IllegalStateException();
    }

    @Override
    public boolean isFinished() {
        return this.state != State.PROCESSING;
    }

    @Override
    public boolean isClosed() {
        return this.state == State.CLOSED;
    }

    @Override
    public void close() {
        this.state = State.CLOSED;
    }

    private static enum State {
        PROCESSING,
        FINISHED,
        CLOSED;

    }
}

